"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.default = filterDuplicateTransactions;
const moment_1 = __importDefault(require("moment"));
const diff_transactions_1 = require("./diff-transactions");
const helpers_1 = require("../helpers");
/*
    This function tries to be smarter in detecting which of the provided
    transactions can safely be used to update the known transaction.
    when a pair of duplicate candidates transaction is detected, the known transaction
    can safely be updated if:
    - the known and provided transactions have all fields equal but the type
    AND one of the below conditions is verified:
    - the known transaction type is unknown and the provided one has a definite type
    - the transaction type has switched from DEFERRED_CARD_TYPE to TRANSACTION_CARD_TYPE,
      and the current date is after the transaction debitDate, as a deferred transaction
      can only become a "normal" transaction once its debit date is passed (a deferred transaction
      is by definition a transaction which debit date is in the future).
*/
function filterDuplicateTransactions(duplicates) {
    const toCreate = [];
    const toUpdate = [];
    const today = new Date();
    for (const [known, provided] of duplicates) {
        // We ignore transactions which differ from more than just the type.
        if (!(0, diff_transactions_1.amountAndLabelAndDateMatch)(known, provided)) {
            toCreate.push(provided);
            continue;
        }
        // We expect the polled/provided transactions to replace any known transaction created by the user
        // or through the recurring transaction system and reset these flags to false, unless both
        // transactions have it enabled.
        const updateBase = {};
        if (!!known.isRecurrentTransaction && !provided.isRecurrentTransaction) {
            updateBase.isRecurrentTransaction = false;
        }
        if (!!known.createdByUser && !provided.createdByUser) {
            updateBase.createdByUser = false;
        }
        // If the type in the database is unknown, set it to the provided one.
        if (known.type === helpers_1.UNKNOWN_TRANSACTION_TYPE && provided.type !== helpers_1.UNKNOWN_TRANSACTION_TYPE) {
            toUpdate.push({
                known,
                update: {
                    ...updateBase,
                    type: provided.type,
                },
            });
            continue;
        }
        // The transaction type which was "deferred_card", is now "card", and the debitDate is now
        // in the past (ie. the change of type is legitimate), update the transaction.
        if (known.debitDate &&
            known.type === helpers_1.DEFERRED_CARD_TYPE.name &&
            provided.type === helpers_1.TRANSACTION_CARD_TYPE.name &&
            (0, moment_1.default)(known.debitDate).isSameOrBefore(today, 'day')) {
            toUpdate.push({
                known,
                update: {
                    ...updateBase,
                    type: helpers_1.TRANSACTION_CARD_TYPE.name,
                },
            });
            continue;
        }
        if (Object.keys(updateBase).length > 0) {
            toUpdate.push({ known, update: updateBase });
            continue;
        }
        // If the known type was set by the user, consider the provided and the
        // known transactions are the same. We ignore the pair.
        if (known.isUserDefinedType) {
            continue;
        }
        // If the provided type is unknown but everything else matches (we already
        // checked that amount, date and label match), do not create it.
        if (known.type !== helpers_1.UNKNOWN_TRANSACTION_TYPE && provided.type === helpers_1.UNKNOWN_TRANSACTION_TYPE) {
            continue;
        }
        toCreate.push(provided);
    }
    return {
        toUpdate,
        toCreate,
    };
}
