﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/sagemaker/SageMakerRequest.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>
#include <aws/sagemaker/model/Tag.h>

#include <utility>

namespace Aws {
namespace SageMaker {
namespace Model {

/**
 */
class CreateModelPackageGroupRequest : public SageMakerRequest {
 public:
  AWS_SAGEMAKER_API CreateModelPackageGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateModelPackageGroup"; }

  AWS_SAGEMAKER_API Aws::String SerializePayload() const override;

  AWS_SAGEMAKER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the model group.</p>
   */
  inline const Aws::String& GetModelPackageGroupName() const { return m_modelPackageGroupName; }
  inline bool ModelPackageGroupNameHasBeenSet() const { return m_modelPackageGroupNameHasBeenSet; }
  template <typename ModelPackageGroupNameT = Aws::String>
  void SetModelPackageGroupName(ModelPackageGroupNameT&& value) {
    m_modelPackageGroupNameHasBeenSet = true;
    m_modelPackageGroupName = std::forward<ModelPackageGroupNameT>(value);
  }
  template <typename ModelPackageGroupNameT = Aws::String>
  CreateModelPackageGroupRequest& WithModelPackageGroupName(ModelPackageGroupNameT&& value) {
    SetModelPackageGroupName(std::forward<ModelPackageGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description for the model group.</p>
   */
  inline const Aws::String& GetModelPackageGroupDescription() const { return m_modelPackageGroupDescription; }
  inline bool ModelPackageGroupDescriptionHasBeenSet() const { return m_modelPackageGroupDescriptionHasBeenSet; }
  template <typename ModelPackageGroupDescriptionT = Aws::String>
  void SetModelPackageGroupDescription(ModelPackageGroupDescriptionT&& value) {
    m_modelPackageGroupDescriptionHasBeenSet = true;
    m_modelPackageGroupDescription = std::forward<ModelPackageGroupDescriptionT>(value);
  }
  template <typename ModelPackageGroupDescriptionT = Aws::String>
  CreateModelPackageGroupRequest& WithModelPackageGroupDescription(ModelPackageGroupDescriptionT&& value) {
    SetModelPackageGroupDescription(std::forward<ModelPackageGroupDescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of key value pairs associated with the model group. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging
   * Amazon Web Services resources</a> in the <i>Amazon Web Services General
   * Reference Guide</i>.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateModelPackageGroupRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateModelPackageGroupRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_modelPackageGroupName;

  Aws::String m_modelPackageGroupDescription;

  Aws::Vector<Tag> m_tags;
  bool m_modelPackageGroupNameHasBeenSet = false;
  bool m_modelPackageGroupDescriptionHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
