﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/mediatailor/MediaTailor_EXPORTS.h>
#include <aws/mediatailor/model/ChannelState.h>
#include <aws/mediatailor/model/ResponseOutputItem.h>
#include <aws/mediatailor/model/SlateSource.h>
#include <aws/mediatailor/model/TimeShiftConfiguration.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace Utils {
namespace Json {
class JsonValue;
}  // namespace Json
}  // namespace Utils
namespace MediaTailor {
namespace Model {
class UpdateChannelResult {
 public:
  AWS_MEDIATAILOR_API UpdateChannelResult() = default;
  AWS_MEDIATAILOR_API UpdateChannelResult(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);
  AWS_MEDIATAILOR_API UpdateChannelResult& operator=(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) associated with the channel.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  UpdateChannelResult& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the channel.</p>
   */
  inline const Aws::String& GetChannelName() const { return m_channelName; }
  template <typename ChannelNameT = Aws::String>
  void SetChannelName(ChannelNameT&& value) {
    m_channelNameHasBeenSet = true;
    m_channelName = std::forward<ChannelNameT>(value);
  }
  template <typename ChannelNameT = Aws::String>
  UpdateChannelResult& WithChannelName(ChannelNameT&& value) {
    SetChannelName(std::forward<ChannelNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Returns the state whether the channel is running or not.</p>
   */
  inline ChannelState GetChannelState() const { return m_channelState; }
  inline void SetChannelState(ChannelState value) {
    m_channelStateHasBeenSet = true;
    m_channelState = value;
  }
  inline UpdateChannelResult& WithChannelState(ChannelState value) {
    SetChannelState(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp of when the channel was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTime() const { return m_creationTime; }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  void SetCreationTime(CreationTimeT&& value) {
    m_creationTimeHasBeenSet = true;
    m_creationTime = std::forward<CreationTimeT>(value);
  }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  UpdateChannelResult& WithCreationTime(CreationTimeT&& value) {
    SetCreationTime(std::forward<CreationTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The slate used to fill gaps between programs in the schedule. You must
   * configure filler slate if your channel uses the <code>LINEAR</code>
   * <code>PlaybackMode</code>. MediaTailor doesn't support filler slate for channels
   * using the <code>LOOP</code> <code>PlaybackMode</code>.</p>
   */
  inline const SlateSource& GetFillerSlate() const { return m_fillerSlate; }
  template <typename FillerSlateT = SlateSource>
  void SetFillerSlate(FillerSlateT&& value) {
    m_fillerSlateHasBeenSet = true;
    m_fillerSlate = std::forward<FillerSlateT>(value);
  }
  template <typename FillerSlateT = SlateSource>
  UpdateChannelResult& WithFillerSlate(FillerSlateT&& value) {
    SetFillerSlate(std::forward<FillerSlateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp that indicates when the channel was last modified.</p>
   */
  inline const Aws::Utils::DateTime& GetLastModifiedTime() const { return m_lastModifiedTime; }
  template <typename LastModifiedTimeT = Aws::Utils::DateTime>
  void SetLastModifiedTime(LastModifiedTimeT&& value) {
    m_lastModifiedTimeHasBeenSet = true;
    m_lastModifiedTime = std::forward<LastModifiedTimeT>(value);
  }
  template <typename LastModifiedTimeT = Aws::Utils::DateTime>
  UpdateChannelResult& WithLastModifiedTime(LastModifiedTimeT&& value) {
    SetLastModifiedTime(std::forward<LastModifiedTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The channel's output properties.</p>
   */
  inline const Aws::Vector<ResponseOutputItem>& GetOutputs() const { return m_outputs; }
  template <typename OutputsT = Aws::Vector<ResponseOutputItem>>
  void SetOutputs(OutputsT&& value) {
    m_outputsHasBeenSet = true;
    m_outputs = std::forward<OutputsT>(value);
  }
  template <typename OutputsT = Aws::Vector<ResponseOutputItem>>
  UpdateChannelResult& WithOutputs(OutputsT&& value) {
    SetOutputs(std::forward<OutputsT>(value));
    return *this;
  }
  template <typename OutputsT = ResponseOutputItem>
  UpdateChannelResult& AddOutputs(OutputsT&& value) {
    m_outputsHasBeenSet = true;
    m_outputs.emplace_back(std::forward<OutputsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of playback mode for this channel.</p> <p> <code>LINEAR</code> -
   * Programs play back-to-back only once.</p> <p> <code>LOOP</code> - Programs play
   * back-to-back in an endless loop. When the last program in the schedule plays,
   * playback loops back to the first program in the schedule.</p>
   */
  inline const Aws::String& GetPlaybackMode() const { return m_playbackMode; }
  template <typename PlaybackModeT = Aws::String>
  void SetPlaybackMode(PlaybackModeT&& value) {
    m_playbackModeHasBeenSet = true;
    m_playbackMode = std::forward<PlaybackModeT>(value);
  }
  template <typename PlaybackModeT = Aws::String>
  UpdateChannelResult& WithPlaybackMode(PlaybackModeT&& value) {
    SetPlaybackMode(std::forward<PlaybackModeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags to assign to the channel. Tags are key-value pairs that you can
   * associate with Amazon resources to help with organization, access control, and
   * cost tracking. For more information, see <a
   * href="https://docs.aws.amazon.com/mediatailor/latest/ug/tagging.html">Tagging
   * AWS Elemental MediaTailor Resources</a>.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  UpdateChannelResult& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  UpdateChannelResult& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tier associated with this Channel.</p>
   */
  inline const Aws::String& GetTier() const { return m_tier; }
  template <typename TierT = Aws::String>
  void SetTier(TierT&& value) {
    m_tierHasBeenSet = true;
    m_tier = std::forward<TierT>(value);
  }
  template <typename TierT = Aws::String>
  UpdateChannelResult& WithTier(TierT&& value) {
    SetTier(std::forward<TierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The time-shifted viewing configuration for the channel. </p>
   */
  inline const TimeShiftConfiguration& GetTimeShiftConfiguration() const { return m_timeShiftConfiguration; }
  template <typename TimeShiftConfigurationT = TimeShiftConfiguration>
  void SetTimeShiftConfiguration(TimeShiftConfigurationT&& value) {
    m_timeShiftConfigurationHasBeenSet = true;
    m_timeShiftConfiguration = std::forward<TimeShiftConfigurationT>(value);
  }
  template <typename TimeShiftConfigurationT = TimeShiftConfiguration>
  UpdateChannelResult& WithTimeShiftConfiguration(TimeShiftConfigurationT&& value) {
    SetTimeShiftConfiguration(std::forward<TimeShiftConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The list of audiences defined in channel.</p>
   */
  inline const Aws::Vector<Aws::String>& GetAudiences() const { return m_audiences; }
  template <typename AudiencesT = Aws::Vector<Aws::String>>
  void SetAudiences(AudiencesT&& value) {
    m_audiencesHasBeenSet = true;
    m_audiences = std::forward<AudiencesT>(value);
  }
  template <typename AudiencesT = Aws::Vector<Aws::String>>
  UpdateChannelResult& WithAudiences(AudiencesT&& value) {
    SetAudiences(std::forward<AudiencesT>(value));
    return *this;
  }
  template <typename AudiencesT = Aws::String>
  UpdateChannelResult& AddAudiences(AudiencesT&& value) {
    m_audiencesHasBeenSet = true;
    m_audiences.emplace_back(std::forward<AudiencesT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetRequestId() const { return m_requestId; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  UpdateChannelResult& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_arn;

  Aws::String m_channelName;

  ChannelState m_channelState{ChannelState::NOT_SET};

  Aws::Utils::DateTime m_creationTime{};

  SlateSource m_fillerSlate;

  Aws::Utils::DateTime m_lastModifiedTime{};

  Aws::Vector<ResponseOutputItem> m_outputs;

  Aws::String m_playbackMode;

  Aws::Map<Aws::String, Aws::String> m_tags;

  Aws::String m_tier;

  TimeShiftConfiguration m_timeShiftConfiguration;

  Aws::Vector<Aws::String> m_audiences;

  Aws::String m_requestId;
  bool m_arnHasBeenSet = false;
  bool m_channelNameHasBeenSet = false;
  bool m_channelStateHasBeenSet = false;
  bool m_creationTimeHasBeenSet = false;
  bool m_fillerSlateHasBeenSet = false;
  bool m_lastModifiedTimeHasBeenSet = false;
  bool m_outputsHasBeenSet = false;
  bool m_playbackModeHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_tierHasBeenSet = false;
  bool m_timeShiftConfigurationHasBeenSet = false;
  bool m_audiencesHasBeenSet = false;
  bool m_requestIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace MediaTailor
}  // namespace Aws
